import { SelectionModel } from '@angular/cdk/collections';
import { Component, OnInit, ViewChild } from '@angular/core';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort } from '@angular/material/sort';
import { MatTableDataSource } from '@angular/material/table';
import { Router } from '@angular/router';
import { NgbModal, NgbModalModule, NgbModule } from '@ng-bootstrap/ng-bootstrap';
import { TranslatePipe } from '@ngx-translate/core';
import { ToastrService } from 'ngx-toastr';
import { Observable, Subject, debounceTime, distinctUntilChanged, switchMap, takeUntil, tap } from 'rxjs';
import Swal from 'sweetalert2';
import { Unsub } from '../../../shared/class/unsub.class';
import { AdminService } from '../../../shared/services/admin/admin.service';
import { LoaderService } from '../../../shared/services/loader.service';
import { TranslationService } from '../../../shared/services/translation.service';

import { TranslatePipes } from '../../../shared/pipes/translate/translate.pipe';
import { MaterialeModules } from '../../../material.module';
import { FormsModule } from '@angular/forms';
import { AdminPageHeaderComponent } from '../../../shared/common/pageheader/admin-page-header/admin-page-header.component';

@Component({
  selector: 'app-department',
  templateUrl: './department.component.html',
  styleUrls: ['./department.component.scss'],
  standalone: true,
  imports: [AdminPageHeaderComponent, TranslatePipes, MaterialeModules, NgbModule, FormsModule],
  providers : [TranslatePipes]
})
export class DepartmentComponent extends Unsub implements OnInit {
  public department: any;
  private searchTerms = new Subject<string>();
  public departmentid: any;
  public departmentname: any;
  public departmentstatus: any = null;
  public sendingResponse: { [key: string]: boolean } = {};

  public refreshToken$: any;
  displayedColumns: string[] = [
    'sno',
    'select',
    'department',
    'status',
    'actions',
  ];
  dataSource!: MatTableDataSource<any>;
  selection = new SelectionModel<any>(true, []);
  permission: any;
  isloader = true;
  @ViewChild(MatPaginator) paginator!: MatPaginator;
  @ViewChild(MatSort) sort!: MatSort;

  constructor(
    private adminService: AdminService,
    private router: Router,
    private toastr: ToastrService,
    private modalService: NgbModal,
    private translate: TranslatePipe,
    private translationService: TranslationService,
    loaderService: LoaderService
  ) {
    /**
    * This code calls the constructor of the parent class in a subclass. 
    * The super() function is used to initialize the parent class and allows access to its properties and methods. 
    * By calling  super(), the subclass can inherit and use the functionality of the parent class.
    **/
    super(loaderService);
    this.translationService.TranslationStatus.subscribe(() => {
      setTimeout(() => {
        this.paginator._intl.itemsPerPageLabel = this.translate.transform('items per page:');
        this.paginator._intl.nextPageLabel = this.translate.transform('next');
        this.paginator._intl.previousPageLabel = this.translate.transform('previous');
        this.paginator._intl.firstPageLabel = this.translate.transform('first page');
        this.paginator._intl.lastPageLabel = this.translate.transform('last page');
        const of = this.translate.transform('of');
        this.paginator._intl.getRangeLabel = (page: number, pageSize: number, length: number) => {
          if (length == 0 || pageSize == 0) return `0 ${of} ${length}`;
          length = Math.max(length, 0);
          const startIndex = page * pageSize;
          const endIndex = startIndex < length ? Math.min(startIndex + pageSize, length) : startIndex + pageSize;

          return `${startIndex + 1} - ${endIndex} ${of} ${length}`;
        };
      }, 100);
    });
    this.translationService.setTranslationaStatus(true);
  }

  ngOnInit(): void {
    this.allDepartments();
    this.adminService.RefreshRquestAdmin$.pipe(
      takeUntil(this.unSubscribe$)
    ).subscribe({
      next: () => {
        this.allDepartments();

      },
    });

    this.searchTerms.pipe(
      debounceTime(300),
      distinctUntilChanged(),
      switchMap((): any => this.loadData())
    ).subscribe();
  }
  allDepartments() {
    this.loaderService.showLoader();
    this.isloader = true;
    this.adminService
      .alldepartments(1, 10, this.filterValue)
      .pipe(takeUntil(this.unSubscribe$))
      .subscribe({
        next: (res) => {
          this.permission = res.permissions;
          this.department = res.queryList;
          this.isloader = false;
          // Assign the data to the data source for the table to render
          this.dataSource = new MatTableDataSource(this.department.data);

          if (this.dataSource) {
            this.paginator.pageSize = this.department.per_page;
            setTimeout(() => {
              this.paginator.pageIndex = this.department.current_page - 1;
              this.paginator.length = this.department.total;
            });
            this.dataSource.paginator = this.paginator;
            
              // this.sort.disableClear = true; // Used to disbale the original soting of data if enabled (Only ASC, DES).
              this.dataSource.sort = this.sort;
              this.dataSource.sortingDataAccessor = (data, sortHeaderId) => {
                switch (sortHeaderId) {
                  case 'user' || 'ticketdetails' || 'status':
                    return this.nestedProperty(data, sortHeaderId);
                  default:
                    return this.nestedCaseInsensitive(data, sortHeaderId);
                }
              }
          }
          if (!this.permission?.includes('Department Delete')) {
            this.displayedColumns = ['sno', 'department', 'status', 'actions'];
          }
          this.loaderService.hideLoader();
        },
        error: () => { this.isloader = true; this.loaderService.hideLoader() }
      });
  }

  filterValue = "";
  applyFilter(event: Event) {
    this.filterValue = (event.target as HTMLInputElement).value;
    this.searchTerms.next(this.filterValue);
  }

  /** Whether the number of selected elements matches the total number of rows. */
  isAllSelected() {
    const numSelected = this.selection.selected.length;
    let numRows;
    if (this.dataSource) {
      numRows = this.dataSource.data.length;
    }
    return numSelected === numRows;
  }
  /** Selects all rows if they are not all selected; otherwise clear selection. */
  toggleAllRows() {
    if (this.isAllSelected()) {
      this.selection.clear();
      return;
    }
    this.selection.select(...this.dataSource.data);
  }

  /** The label for the checkbox on the passed row */
  checkboxLabel(row?: any): string {
    if (!row) {
      return `${this.isAllSelected() ? 'deselect' : 'select'} all`;
    }
    return `${this.selection.isSelected(row) ? 'deselect' : 'select'} row ${row?.id + 1
      }`;
  }

  addModelPopup(content: any) {
    const modalRef = this.modalService.open(content, { backdrop: 'static', keyboard: false });
    modalRef.result.then(
      () => {
      },
      () => {
      }
    );
  }
  editModelPopup(content: any, row: any) {

    this.departmentid = row?.id;
    this.departmentname = row?.departmentname;
    this.departmentstatus = row?.status;
    const modalRef = this.modalService.open(content, { size: 'lg', backdrop: 'static', keyboard: false });
    modalRef.result.then(
      () => {
        this.departmentid = '';
        this.departmentname = '';
        this.departmentstatus = '';
      },
      () => {
        this.departmentid = '';
        this.departmentname = '';
        this.departmentstatus = '';
      }
    );
  }

  addDepartment(ngForm: any, buttonId) {
    if (ngForm.valid) {
      this.sendingResponse[buttonId] = true;

      const status = ngForm.value?.status == '' ? 0 : 1;
      const data = {

        id: ngForm.value?.id,
        departmentname: ngForm.value?.departmentname,
        status: status,
      };
      this.adminService
        .createDepartment(data)
        .pipe(takeUntil(this.unSubscribe$))
        .subscribe({
          next: (res) => {
            this.sendingResponse[buttonId] = false;
            if (res.success) {
              this.modalService.dismissAll();
              this.toastr.success(this.translate.transform(res.success));
            }
          },
          error: () => {
            this.sendingResponse[buttonId] = false;
          },
        });
    }
  }
  editDepartment(ngForm: any, buttonId) {
    if (ngForm.valid) {
      this.sendingResponse[buttonId] = true;
      const data = {

        id: ngForm.value?.id,
        departmentname: ngForm.value?.departmentname,
        status: ngForm.value?.status,
      };
      this.adminService
        .createDepartment(data)
        .pipe(takeUntil(this.unSubscribe$))
        .subscribe({
          next: (res) => {
            this.sendingResponse[buttonId] = false;
            if (res.success) {
              this.modalService.dismissAll();
              this.toastr.success(this.translate.transform(res.success));
            }
          },
          error: () => {
            this.sendingResponse[buttonId] = false;
          },
        });
    }
  }
  deleteDepartment(row: any, buttonId) {
    const data = {

      id: row?.id,
    };
    Swal.fire({
      icon: 'warning',
      title: this.translate.transform('Are you sure ?'),
      text: this.translate.transform('Your will not be able to recover!'),
      showCancelButton: true,
      confirmButtonColor: '#6259ca',
      cancelButtonColor: '#f11541',
      confirmButtonText: this.translate.transform('Yes, delete it!'),
      cancelButtonText: this.translate.transform('Close'),
      reverseButtons: true,
    }).then((result: any) => {
      if (result.isConfirmed) {
        this.sendingResponse[buttonId] = true;
        this.adminService
          .deleteDepartment(row?.id, data)
          .pipe(takeUntil(this.unSubscribe$))
          .subscribe({
            next: (res) => {
              this.sendingResponse[buttonId] = false;
              if (res.success) {
                this.toastr.success(this.translate.transform(res.success));
              }
            },
            error: () => {
              this.sendingResponse[buttonId] = false;
            },
          });
      }
    });
  }
  massDeleteDepartments(buttonId) {
    if (!this.selection.selected.length) {
      this.toastr.error(this.translate.transform('Please select atleast one Department'));
      return;
    }
    const id: any = [];
    this.selection.selected.map(function (ele) {
      id.push(ele.id);
    });
    const data = {

      id: id,
    };
    Swal.fire({
      icon: 'warning',
      title: this.translate.transform('Are you sure ?'),
      text: this.translate.transform('Your will not be able to recover!'),
      showCancelButton: true,
      confirmButtonColor: '#6259ca',
      cancelButtonColor: '#f11541',
      confirmButtonText: this.translate.transform('Yes, delete it!'),
      cancelButtonText: this.translate.transform('Close'),
      reverseButtons: true,
    }).then((result: any) => {
      if (result.isConfirmed) {
        this.sendingResponse[buttonId] = true;
        this.adminService
          .deleteAllDepartment(data)
          .pipe(takeUntil(this.unSubscribe$))
          .subscribe({
            next: (res) => {
              this.sendingResponse[buttonId] = false;
              this.selection.clear();
              if (res.success) {
                this.toastr.success(this.translate.transform(res.success));
              }
            },
            error: () => {
              this.sendingResponse[buttonId] = false;
            },
          });
      }
    });
  }
  departmentsStatusChange(row) {
    const val = row?.status !== 1 ? 1 : 0;

    const data = {

      status: val,
    };
    this.adminService
      .statuschangeDepartment(row?.id, data)
      .pipe(takeUntil(this.unSubscribe$))
      .subscribe({
        next: (res) => {
          if (res.success) {
            this.toastr.success(this.translate.transform(res.success));
          }
        }
      });
  }

  loadData(): Observable<any> {
    this.isloader = true;
    return this.adminService.alldepartments(this.paginator.pageIndex + 1, this.paginator.pageSize, this.filterValue).pipe(
      tap((data: any) => {
        this.isloader = false;
        this.dataSource.data = data.queryList.data;
        setTimeout(() => {
          this.paginator.pageIndex = data.queryList.current_page - 1;
          this.paginator.length = data.queryList.total;
          this.paginator.pageSize = data.queryList.per_page;
        });
      }));
  }

  onPaginateTable() {
    this.selection.clear();
    this.loadData().subscribe();
  }
  // this is needed to allow sorting on nested properties
  nestedProperty = (data: any, sortHeaderId: string): string | number => {
    return sortHeaderId
      ?.split('.')
      .reduce((accumulator, key) => accumulator && accumulator[key], data) as
      | string
      | number;
  };
  // this is needed to have caseInsensitive sorting
  caseInsensitive = (data: any, sortHeaderId: string): string | number => {
    const value = data[sortHeaderId];
    return typeof value === 'string' ? value.toUpperCase() :
      value;
  };
  nestedCaseInsensitive = (data: any, sortHeaderId: string): string | number => {
    const value = sortHeaderId
      ?.split('.')
      .reduce((accumulator, key) => accumulator && accumulator[key], data) as
      | string
      | number;
    return typeof value === 'string' ? value.toUpperCase() : value;
  };
}
