import { CommonModule } from '@angular/common';
import { Component, EventEmitter, inject, Input, Output, SimpleChanges } from '@angular/core';
import { ControlContainer, FormControl, FormGroup, ReactiveFormsModule, ValidatorFn, Validators } from '@angular/forms';
import { SharedModule } from '../../../../shared/shared.module';

@Component({
  selector: 'app-textarea-input',
  standalone: true,
  imports: [ReactiveFormsModule, CommonModule, SharedModule],
  viewProviders:[
    {
      provide: ControlContainer,
      useFactory : ()=>inject(ControlContainer, { skipSelf : true })
    }
  ],
  template: `
  <textarea type="text" [ngClass]="classText" [id]="idField ? idField : nameField" [formControlName]="nameField" [placeholder]="placeholder" autocomplete="off" rows="3"
      (input)="onTextareaInput($event.target)" (keyup)="keyupFn()" >{{ value }}</textarea>
  <div *ngIf="formControl.invalid && formControl.touched" class="error-invalid">
    <span class="text-danger" *ngIf="formControl.hasError('required')">{{'This field is required' | translate}}</span>
  </div>`
})
export class TextareaInputComponent {
  @Input() nameField!: string;
  @Input() value!: string;
  @Input() idField!: string;
  @Input() classText!: string;
  @Input() placeholder!: string;
  @Input() validationRules!: any[];
  @Output() valueChange: EventEmitter<string> = new EventEmitter<string>();
  parentContainer = inject(ControlContainer);

  get parentFormGroup(){
    return this.parentContainer.control as FormGroup;
  }
  get formControl(): FormControl {
    return this.parentFormGroup.get(this.nameField) as FormControl;
  }

  ngOnInit() {
    const validators: ValidatorFn[] = [];
    if (this.validationRules.includes('required')) {
      validators.push(Validators.required);
    }
    this.parentFormGroup.addControl(this.nameField, new FormControl(this.value ?? '', validators))
  }

  keyupFn(){
  }
  ngOnChanges(changes: SimpleChanges): void {    
    // Check if the initialSelect input has changed
    if (changes['value']) {
      // Update the form control value
      this.updateFormControl();
    }
  }

  private updateFormControl(): void {
    if (this.parentFormGroup) {
      // Update the value of the form control
      const control = this.parentFormGroup.get(this.nameField);
      if (control) {
        control.setValue(this.value ?? '');
        control.valueChanges.subscribe((value: string) => {
          this.valueChange.emit(value);
        });
        const validators: ValidatorFn[] = [];
        if (this.validationRules.includes('required')) {
          validators.push(Validators.required);
        }
        control.setValidators(validators);
        control.updateValueAndValidity();
      } 
    }
  }
  onTextareaInput(target: any) {
    if(target?.value){
      this.valueChange.emit(target.value);
    }
  }
  ngOnDestroy() {
    this.parentFormGroup.removeControl(this.nameField);
  }
}
